<?php
namespace App\Models;

use App\Database\Database;
use PDO;

class User {
    private PDO $pdo;

    public function __construct() {
        $config = require __DIR__ . '/../../config/config.php';
        $dbConfig = $config['db'];
        $this->pdo = Database::getInstance($dbConfig)->getConnection();
    }

    public function createUser(string $username, string $password, string $full_name, string $email, string $role, ?int $branch_id = null): bool {
        $password_hash = password_hash($password, PASSWORD_DEFAULT);
        $sql = "INSERT INTO users (username, password_hash, full_name, email, role, branch_id) VALUES (?, ?, ?, ?, ?, ?)";
        $stmt = $this->pdo->prepare($sql);
        return $stmt->execute([$username, $password_hash, $full_name, $email, $role, $branch_id]);
    }

    public function authenticate(string $username, string $password) {
        $sql = "SELECT * FROM users WHERE username = ?";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([$username]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        if ($user && password_verify($password, $user['password_hash'])) {
            unset($user['password_hash']);
            return $user;
        }
        return false;
    }

    public function countUsers(): int {
        $sql = "SELECT COUNT(*) FROM users";
        $stmt = $this->pdo->query($sql);
        return (int)$stmt->fetchColumn();
    }

    public function getAllUsers(): array {
        $stmt = $this->pdo->query("SELECT user_id, username, email, full_name, role, branch_id FROM users ORDER BY user_id DESC");
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getUserById(int $user_id): ?array {
        $stmt = $this->pdo->prepare("SELECT * FROM users WHERE user_id = ?");
        $stmt->execute([$user_id]);
        return $stmt->fetch(PDO::FETCH_ASSOC) ?: null;
    }

    public function updateUser(int $user_id, string $username, string $full_name, string $email, string $role, ?int $branch_id = null): bool {
        $sql = "UPDATE users SET username = ?, full_name = ?, email = ?, role = ?, branch_id = ? WHERE user_id = ?";
        $stmt = $this->pdo->prepare($sql);
        return $stmt->execute([$username, $full_name, $email, $role, $branch_id, $user_id]);
    }

    public function deleteUser(int $user_id): bool {
        $stmt = $this->pdo->prepare("DELETE FROM users WHERE user_id = ?");
        return $stmt->execute([$user_id]);
    }

    /**
     * Get users who are sales representatives.
     * Assumes your users table has a 'role' column with 'sales_rep' for sales reps.
     * @return array List of sales reps with user_id and full_name
     */
    public function getSalesReps(): array {
        $sql = "SELECT user_id, full_name FROM users WHERE role = 'sales_rep' ORDER BY full_name";
        $stmt = $this->pdo->query($sql);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
}
