<?php
namespace App\Models;

use App\Database\Database;

class StockMovement {
    private $pdo;

    public function __construct() {
        $config = require __DIR__ . '/../../config/config.php';
        $dbConfig = $config['db'];
        $this->pdo = Database::getInstance($dbConfig)->getConnection();
    }

    // Get current stock grouped by product and branch by summing movements
    public function getCurrentStock(): array {
        $sql = "
            SELECT 
                i.inventory_id,
                i.product_name,
                i.sku,
                b.branch_id,
                b.branch_name,
                COALESCE(SUM(sm.quantity), 0) as current_quantity
            FROM inventory i
            LEFT JOIN stock_movements sm ON i.inventory_id = sm.product_id
            LEFT JOIN branches b ON sm.branch_id = b.branch_id
            GROUP BY i.inventory_id, b.branch_id
            ORDER BY i.product_name
        ";
        $stmt = $this->pdo->query($sql);
        return $stmt->fetchAll(\PDO::FETCH_ASSOC);
    }

    // Add a stock movement record (in, out, adjustment)
    public function addMovement(int $product_id, ?int $branch_id, int $quantity, string $movement_type, string $reference = ''): bool {
        $sql = "INSERT INTO stock_movements (product_id, branch_id, quantity, movement_type, reference, created_at) VALUES (?, ?, ?, ?, ?, NOW())";
        $stmt = $this->pdo->prepare($sql);
        return $stmt->execute([$product_id, $branch_id, $quantity, $movement_type, $reference]);
    }

    // Get a movement by ID
    public function getMovementById(int $movement_id): ?array {
        $stmt = $this->pdo->prepare("SELECT * FROM stock_movements WHERE stock_movement_id = ?");
        $stmt->execute([$movement_id]);
        return $stmt->fetch(\PDO::FETCH_ASSOC) ?: null;
    }

    // Update a stock movement record
    public function updateMovement(int $movement_id, int $product_id, ?int $branch_id, int $quantity, string $movement_type, string $reference = ''): bool {
        $sql = "UPDATE stock_movements SET product_id = ?, branch_id = ?, quantity = ?, movement_type = ?, reference = ? WHERE stock_movement_id = ?";
        $stmt = $this->pdo->prepare($sql);
        return $stmt->execute([$product_id, $branch_id, $quantity, $movement_type, $reference, $movement_id]);
    }

    // Delete a stock movement record
    public function deleteMovement(int $movement_id): bool {
        $stmt = $this->pdo->prepare("DELETE FROM stock_movements WHERE stock_movement_id = ?");
        return $stmt->execute([$movement_id]);
    }

    // Get movements with applied filters
    public function getFilteredMovements(array $filters = []): array {
        $query = "
            SELECT sm.*, i.product_name, i.sku, b.branch_name 
            FROM stock_movements sm
            LEFT JOIN inventory i ON sm.product_id = i.inventory_id
            LEFT JOIN branches b ON sm.branch_id = b.branch_id
            WHERE 1=1
        ";
        $params = [];

        if (!empty($filters['product_id'])) {
            $query .= " AND sm.product_id = ?";
            $params[] = $filters['product_id'];
        }
        if (!empty($filters['branch_id'])) {
            $query .= " AND sm.branch_id = ?";
            $params[] = $filters['branch_id'];
        }
        if (!empty($filters['movement_type'])) {
            $query .= " AND sm.movement_type = ?";
            $params[] = $filters['movement_type'];
        }
        if (!empty($filters['date_from'])) {
            $query .= " AND sm.created_at >= ?";
            $params[] = $filters['date_from'] . " 00:00:00";
        }
        if (!empty($filters['date_to'])) {
            $query .= " AND sm.created_at <= ?";
            $params[] = $filters['date_to'] . " 23:59:59";
        }

        $query .= " ORDER BY sm.created_at DESC";

        $stmt = $this->pdo->prepare($query);
        $stmt->execute($params);
        return $stmt->fetchAll(\PDO::FETCH_ASSOC);
    }
    
    public function getRecentMovements(int $limit = 5): array {
    $sql = "
        SELECT sm.*, i.product_name, i.sku, b.branch_name
        FROM stock_movements sm
        LEFT JOIN inventory i ON sm.product_id = i.inventory_id
        LEFT JOIN branches b ON sm.branch_id = b.branch_id
        ORDER BY sm.created_at DESC
        LIMIT :limit
    ";
    $stmt = $this->pdo->prepare($sql);
    $stmt->bindValue(':limit', $limit, \PDO::PARAM_INT);
    $stmt->execute();
    return $stmt->fetchAll(\PDO::FETCH_ASSOC);
 }


    // List all movements without filters
    public function getAllMovements(): array {
        $stmt = $this->pdo->query("
            SELECT sm.*, i.product_name, i.sku, b.branch_name 
            FROM stock_movements sm
            LEFT JOIN inventory i ON sm.product_id = i.inventory_id
            LEFT JOIN branches b ON sm.branch_id = b.branch_id
            ORDER BY sm.created_at DESC
        ");
        return $stmt->fetchAll(\PDO::FETCH_ASSOC);
    }
}
