<?php

namespace App\Models;

require_once __DIR__ . '/../Helpers/helpers.php';

use App\Database\Database;
use PDO;
use Exception;

class Sale {
    private PDO $pdo;
    private Product $productModel;

    public function __construct() {
        $config = require __DIR__ . '/../../config/config.php';
        $dbConfig = $config['db'];
        $this->pdo = Database::getInstance($dbConfig)->getConnection();
        $this->productModel = new Product();
    }

    public function createSale(array $saleData, array $items): int {
        try {
            $this->pdo->beginTransaction();

            $sql = "INSERT INTO sales (total_amount, payment_method, sales_rep_id, branch_id, created_at) VALUES (?, ?, ?, ?, NOW())";
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute([
                $saleData['total_amount'],
                $saleData['payment_method'],
                $saleData['sales_rep_id'],
                $saleData['branch_id']
            ]);
            $saleId = (int)$this->pdo->lastInsertId();

            $sqlItem = "INSERT INTO sale_items (sale_id, product_id, quantity, unit_price, unit) VALUES (?, ?, ?, ?, ?)";
            $stmtItem = $this->pdo->prepare($sqlItem);

            foreach ($items as $item) {
                $stmtItem->execute([
                    $saleId,
                    $item['product_id'],
                    $item['quantity'],
                    $item['unit_price'],
                    $item['unit']
                ]);

                $product = $this->productModel->getById($item['product_id']);
                if (!$product) {
                    throw new Exception("Product ID {$item['product_id']} not found.");
                }
                $newStock = $product['stock_quantity'] - $item['quantity'];
                if ($newStock < 0) {
                    throw new Exception("Insufficient stock for product ID {$item['product_id']}.");
                }
                $this->productModel->updateStock($item['product_id'], $newStock);
            }

            $this->pdo->commit();
            return $saleId;
        } catch (Exception $e) {
            $this->pdo->rollBack();
            throw $e;
        }
    }

    public function getSaleById(int $saleId): ?array {
        $stmt = $this->pdo->prepare("SELECT * FROM sales WHERE sale_id = ?");
        $stmt->execute([$saleId]);
        $sale = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$sale) {
            return null;
        }

        $stmtItems = $this->pdo->prepare(
            "SELECT si.*, p.product_name 
             FROM sale_items si 
             JOIN products p ON si.product_id = p.product_id 
             WHERE si.sale_id = ?"
        );
        $stmtItems->execute([$saleId]);
        $items = $stmtItems->fetchAll(PDO::FETCH_ASSOC);

        $sale['items'] = $items;
        return $sale;
    }

    public function getSalesByRep(int $salesRepId, ?string $startDate = null, ?string $endDate = null): array {
        $params = ['sales_rep_id' => $salesRepId];
        $sql = "SELECT s.*, si.product_id, si.quantity, si.unit_price, u.unit_name, p.product_name
        FROM sales s
        JOIN sale_items si ON s.sale_id = si.sale_id
        JOIN products p ON si.product_id = p.product_id
        LEFT JOIN units u ON si.unit_id = u.unit_id
        WHERE s.sales_rep_id = :sales_rep_id";


        if ($startDate) {
            $sql .= " AND s.created_at >= :start_date";
            $params['start_date'] = $startDate;
        }
        if ($endDate) {
            $sql .= " AND s.created_at <= :end_date";
            $params['end_date'] = $endDate;
        }

        $sql .= " ORDER BY s.created_at DESC";

        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function saveSale(array $saleData): bool {
        if (empty($saleData['sale_ref'])) {
            throw new Exception('sale_ref is required');
        }
        $sql = "INSERT INTO sales (sale_ref, total_amount, payment_method, sales_rep_id, branch_id, created_at)
                VALUES (:sale_ref, :total_amount, :payment_method, :sales_rep_id, :branch_id, :created_at)
                ON DUPLICATE KEY UPDATE
                total_amount = VALUES(total_amount),
                payment_method = VALUES(payment_method),
                sales_rep_id = VALUES(sales_rep_id),
                branch_id = VALUES(branch_id),
                created_at = VALUES(created_at)";
        $stmt = $this->pdo->prepare($sql);
        $executed = $stmt->execute([
            ':sale_ref' => $saleData['sale_ref'],
            ':total_amount' => $saleData['total_amount'],
            ':payment_method' => $saleData['payment_method'],
            ':sales_rep_id' => $saleData['sales_rep_id'],
            ':branch_id' => $saleData['branch_id'],
            ':created_at' => $saleData['created_at'],
        ]);
        if (!$executed) {
            throw new Exception('Failed to save sale header');
        }
        $saleId = (int) $this->pdo->lastInsertId();
        if ($saleId === 0) {
            $stmtFind = $this->pdo->prepare("SELECT sale_id FROM sales WHERE sale_ref = ?");
            $stmtFind->execute([$saleData['sale_ref']]);
            $existingSale = $stmtFind->fetch(PDO::FETCH_ASSOC);
            $saleId = $existingSale['sale_id'] ?? 0;
        }
        if ($saleId === 0) {
            throw new Exception('Failed to determine sale ID');
        }
        $stmtDelete = $this->pdo->prepare("DELETE FROM sale_items WHERE sale_id = ?");
        $stmtDelete->execute([$saleId]);
        $stmtItem = $this->pdo->prepare("INSERT INTO sale_items (sale_id, product_id, quantity, unit_price, unit) VALUES (?, ?, ?, ?, ?)");
        foreach ($saleData['items'] as $item) {
            $stmtItem->execute([
                $saleId,
                $item['product_id'],
                $item['quantity'],
                $item['unit_price'],
                $item['unit'],
            ]);
        }
        return true;
    }

    public function getTotalSales(string $date): float {
        $sql = "SELECT IFNULL(SUM(total_amount), 0) AS total FROM sales WHERE DATE(created_at) = ?";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([$date]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return (float) ($row['total'] ?? 0);
    }

    public function getNewOrdersCount(): int {
        $sql = "SELECT COUNT(*) FROM sales WHERE DATE(created_at) = CURDATE()";
        return (int) $this->pdo->query($sql)->fetchColumn();
    }

    public function getMostSoldProducts(string $date, int $limit = 5): array {
        $limit = (int)$limit;
        $sql = "SELECT p.product_name, SUM(si.quantity) AS quantity
                FROM sale_items si
                JOIN sales s ON s.sale_id = si.sale_id
                JOIN products p ON p.product_id = si.product_id
                WHERE DATE(s.created_at) = ?
                GROUP BY si.product_id
                ORDER BY quantity DESC
                LIMIT $limit";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([$date]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getDailyRevenue(string $date): float {
        return $this->getTotalSales($date);
    }

    public function getAllSales(): array {
        $stmt = $this->pdo->query("SELECT * FROM sales ORDER BY created_at DESC");
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getTopSales(int $limit = 5): array {
        $sql = "SELECT * FROM sales WHERE DATE(created_at) = CURDATE() ORDER BY total_amount DESC LIMIT ?";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([$limit]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
}
