<?php
namespace App\Models;

use App\Database\Database;
use PDO;
use Exception;

class PurchaseOrder {
    private PDO $pdo;

    public function __construct() {
        $config = require __DIR__ . '/../../config/config.php';
        $dbConfig = $config['db'];
        $this->pdo = Database::getInstance($dbConfig)->getConnection();
    }

    /**
     * Create Purchase Order with items
     * @param int $supplierId
     * @param int $branchId
     * @param string $orderDate (Y-m-d)
     * @param array $items Array of ['product_id'=>int, 'quantity'=>int, 'unit_price'=>float]
     * @param string|null $status
     * @return int Inserted purchase order ID
     * @throws Exception
     */
    public function createPurchaseOrder(int $supplierId, int $branchId, string $orderDate, array $items, string $status = 'Pending'): int {
        try {
            $this->pdo->beginTransaction();

            // Calculate total amount
            $totalAmount = 0;
            foreach ($items as $item) {
                $totalAmount += $item['quantity'] * $item['unit_price'];
            }

            $sql = "INSERT INTO purchase_orders (supplier_id, branch_id, order_date, status, total_amount) VALUES (:supplier_id, :branch_id, :order_date, :status, :total_amount)";
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute([
                ':supplier_id' => $supplierId,
                ':branch_id' => $branchId,
                ':order_date' => $orderDate,
                ':status' => $status,
                ':total_amount' => $totalAmount
            ]);
            $poId = (int)$this->pdo->lastInsertId();

            $itemSql = "INSERT INTO po_items (po_id, product_id, quantity, unit_price) VALUES (:po_id, :product_id, :quantity, :unit_price)";
            $itemStmt = $this->pdo->prepare($itemSql);
            foreach ($items as $item) {
                $itemStmt->execute([
                    ':po_id' => $poId,
                    ':product_id' => $item['product_id'],
                    ':quantity' => $item['quantity'],
                    ':unit_price' => $item['unit_price'],
                ]);
            }

            $this->pdo->commit();

            return $poId;
        } catch (Exception $ex) {
            $this->pdo->rollBack();
            throw $ex;
        }
    }

    /**
     * Get a purchase order with its items by ID
     * @param int $poId
     * @return array|null
     */
    public function getPurchaseOrderById(int $poId): ?array {
        $sql = "SELECT po.*, s.name AS supplier_name, b.branch_name FROM purchase_orders po
                LEFT JOIN suppliers s ON po.supplier_id = s.supplier_id
                LEFT JOIN branches b ON po.branch_id = b.branch_id
                WHERE po.po_id = :po_id";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([':po_id' => $poId]);
        $po = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$po) {
            return null;
        }

        $itemsSql = "SELECT pi.*, p.product_name FROM po_items pi
                    LEFT JOIN products p ON pi.product_id = p.product_id
                    WHERE pi.po_id = :po_id";
        $itemsStmt = $this->pdo->prepare($itemsSql);
        $itemsStmt->execute([':po_id' => $poId]);
        $items = $itemsStmt->fetchAll(PDO::FETCH_ASSOC);

        $po['items'] = $items;
        return $po;
    }

    // Additional methods can include updatePOStatus, listPOs with filters & pagination, deletePO etc.
}
