<?php
namespace App\Models;

use App\Database\Database;
use PDO;
use Exception;

class Product {
    private \PDO $pdo;

    public function __construct() {
        $config = require __DIR__ . '/../../config/config.php';
        $dbConfig = $config['db'];
        $this->pdo = Database::getInstance($dbConfig)->getConnection();
    }

    // Fetch all products with joins to units, categories, branches for display
    public function getAll(): array {
        $sql = "SELECT p.*, u.unit_name, c.category_name, b.branch_name
            FROM products p
            LEFT JOIN units u ON p.unit_id = u.unit_id
            LEFT JOIN categories c ON p.category_id = c.category_id
            LEFT JOIN branches b ON p.branch_id = b.branch_id
            ORDER BY p.product_name";
        $stmt = $this->pdo->query($sql);
        return $stmt->fetchAll(\PDO::FETCH_ASSOC);
    }

    // Fetch a single product by ID
    public function getById(int $id): ?array {
        $sql = "SELECT * FROM products WHERE product_id = ?";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([$id]);
        $product = $stmt->fetch(\PDO::FETCH_ASSOC);
        return $product ?: null;
    }

    // Create product - expects associative array with keys matching table columns (use strict keys)
    public function create(array $data): int|false {
        $sql = "INSERT INTO products 
            (product_name, unit_id, wholesale_price, retail_price, barcode, description, category_id, stock_quantity, expiry_date, batch_number, branch_id, is_pharma)
            VALUES (:product_name, :unit_id, :wholesale_price, :retail_price, :barcode, :description, :category_id, :stock_quantity, :expiry_date, :batch_number, :branch_id, :is_pharma)";
        $stmt = $this->pdo->prepare($sql);

        $params = [
            ':product_name' => $data['product_name'] ?? '',
            ':unit_id' => $data['unit_id'] ?? null,
            ':wholesale_price' => $data['wholesale_price'] ?? 0.0,
            ':retail_price' => $data['retail_price'] ?? 0.0,
            ':barcode' => $data['barcode'] ?? null,
            ':description' => $data['description'] ?? null,
            ':category_id' => $data['category_id'] ?? null,
            ':stock_quantity' => $data['stock_quantity'] ?? 0,
            ':expiry_date' => $data['expiry_date'] ?? null,
            ':batch_number' => $data['batch_number'] ?? null,
            ':branch_id' => $data['branch_id'] ?? null,
            ':is_pharma' => $data['is_pharma'] ?? 0
        ];

        if ($stmt->execute($params)) {
            return (int)$this->pdo->lastInsertId();
        }
        return false;
    }

    // Update product by ID with given data array
    public function update(int $id, array $data): bool {
        $fields = [];
        $params = [];
        foreach ($data as $key => $value) {
            $fields[] = "$key = :$key";
            $params[":$key"] = $value;
        }
        $params[':product_id'] = $id;

        $sql = "UPDATE products SET " . implode(', ', $fields) . " WHERE product_id = :product_id";
        $stmt = $this->pdo->prepare($sql);
        return $stmt->execute($params);
    }

    // Delete product by ID
    public function delete(int $id): bool {
        $stmt = $this->pdo->prepare("DELETE FROM products WHERE product_id = ?");
        return $stmt->execute([$id]);
    }

    /**
     * Get products with stock_quantity below a defined threshold.
     * @param int $threshold minimum stock level to trigger low stock alert (default 10)
     * @return array
     */
    public function getLowStockProducts(int $threshold = 10): array {
        $sql = "SELECT p.*, u.unit_name, c.category_name 
                FROM products p
                LEFT JOIN units u ON p.unit_id = u.unit_id
                LEFT JOIN categories c ON p.category_id = c.category_id
                WHERE p.stock_quantity <= ?
                ORDER BY p.stock_quantity ASC";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([$threshold]);
        return $stmt->fetchAll(\PDO::FETCH_ASSOC);
    }

    /**
     * Count total products in the system
     * @return int
     */
    public function countAll(): int {
        $sql = "SELECT COUNT(*) FROM products";
        return (int) $this->pdo->query($sql)->fetchColumn();
    }
}
