<?php
// src/Controllers/UserController.php

require_once __DIR__ . '/../Models/User.php';
require_once __DIR__ . '/../Middleware/AuthMiddleware.php';

class UserController {
    private $userModel;
    private $authMiddleware;

    public function __construct() {
        $this->userModel = new User();
        $this->authMiddleware = new AuthMiddleware();
    }

    // List all users
    public function listUsers() {
        $this->authMiddleware->handle(function() {
            header('Content-Type: application/json');

            $users = $this->userModel->getAllUsers(); // Implement this method in User model
            echo json_encode(['users' => $users]);
        });
    }

    // Get a single user by ID
    public function getUser($userId) {
        $this->authMiddleware->handle(function() use ($userId) {
            header('Content-Type: application/json');
            $user = $this->userModel->getUserById($userId);
            if ($user) {
                echo json_encode(['user' => $user]);
            } else {
                http_response_code(404);
                echo json_encode(['error' => 'User not found']);
            }
        });
    }

    // Create a new user
    public function createUser() {
        $this->authMiddleware->handle(function() {
            header('Content-Type: application/json');

            if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
                http_response_code(405);
                echo json_encode(['error' => 'Method Not Allowed']);
                exit;
            }

            $input = json_decode(file_get_contents('php://input'), true);

            if (empty($input['username']) || empty($input['password']) || empty($input['full_name']) || empty($input['email']) || empty($input['role'])) {
                http_response_code(400);
                echo json_encode(['error' => 'Required fields missing']);
                exit;
            }

            $result = $this->userModel->createUser(
                $input['username'],
                $input['password'],
                $input['full_name'],
                $input['email'],
                $input['role'],
                $input['branch_id'] ?? null
            );

            if ($result) {
                http_response_code(201);
                echo json_encode(['success' => true]);
            } else {
                http_response_code(500);
                echo json_encode(['error' => 'User creation failed']);
            }
        });
    }

    // Update existing user (excluding password)
    public function updateUser($userId) {
        $this->authMiddleware->handle(function() use ($userId) {
            header('Content-Type: application/json');
            if ($_SERVER['REQUEST_METHOD'] !== 'PUT' && $_SERVER['REQUEST_METHOD'] !== 'PATCH') {
                http_response_code(405);
                echo json_encode(['error' => 'Method Not Allowed']);
                exit;
            }
            $input = json_decode(file_get_contents('php://input'), true);
            $updated = $this->userModel->updateUser($userId, $input);
            if ($updated) {
                echo json_encode(['success' => true]);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'Update failed or no changes made']);
            }
        });
    }

    // Delete user by ID
    public function deleteUser($userId) {
        $this->authMiddleware->handle(function() use ($userId) {
            header('Content-Type: application/json');
            if ($_SERVER['REQUEST_METHOD'] !== 'DELETE') {
                http_response_code(405);
                echo json_encode(['error' => 'Method Not Allowed']);
                exit;
            }
            $deleted = $this->userModel->deleteUser($userId);
            if ($deleted) {
                echo json_encode(['success' => true]);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'Delete failed']);
            }
        });
    }
}
