<?php
// src/Controllers/ProductController.php

require_once __DIR__ . '/../Models/Product.php';
require_once __DIR__ . '/../Middleware/AuthMiddleware.php';

class ProductController {
    private $productModel;
    private $authMiddleware;

    public function __construct() {
        $this->productModel = new Product();
        $this->authMiddleware = new AuthMiddleware();
    }

    // List all products (with optional filters)
    public function listProducts() {
        $this->authMiddleware->handle(function() {
            header('Content-Type: application/json');

            $products = $this->productModel->getAll();
            echo json_encode(['products' => $products]);
        });
    }

    // Get details of a single product by ID
    public function getProduct($id) {
        $this->authMiddleware->handle(function() use ($id) {
            header('Content-Type: application/json');

            $product = $this->productModel->getById($id);
            if ($product) {
                echo json_encode(['product' => $product]);
            } else {
                http_response_code(404);
                echo json_encode(['error' => 'Product not found']);
            }
        });
    }

    // Create a new product entry
    public function createProduct() {
        $this->authMiddleware->handle(function() {
            header('Content-Type: application/json');

            if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
                http_response_code(405);
                echo json_encode(['error' => 'Method Not Allowed']);
                exit;
            }

            $input = json_decode(file_get_contents('php://input'), true);

            // Basic validation - expand as needed
            if (empty($input['product_name']) || !isset($input['retail_price'])) {
                http_response_code(400);
                echo json_encode(['error' => 'Product name and retail price are required']);
                exit;
            }

            $result = $this->productModel->create($input);
            if ($result) {
                http_response_code(201);
                echo json_encode(['success' => true, 'product_id' => $result]);
            } else {
                http_response_code(500);
                echo json_encode(['error' => 'Failed to create product']);
            }
        });
    }

    // Update existing product
    public function updateProduct($id) {
        $this->authMiddleware->handle(function() use ($id) {
            header('Content-Type: application/json');

            if ($_SERVER['REQUEST_METHOD'] !== 'PUT' && $_SERVER['REQUEST_METHOD'] !== 'PATCH') {
                http_response_code(405);
                echo json_encode(['error' => 'Method Not Allowed']);
                exit;
            }

            $input = json_decode(file_get_contents('php://input'), true);

            $updated = $this->productModel->update($id, $input);
            if ($updated) {
                echo json_encode(['success' => true]);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'Failed to update product or no changes made']);
            }
        });
    }

    // Delete product by ID
    public function deleteProduct($id) {
        $this->authMiddleware->handle(function() use ($id) {
            header('Content-Type: application/json');

            if ($_SERVER['REQUEST_METHOD'] !== 'DELETE') {
                http_response_code(405);
                echo json_encode(['error' => 'Method Not Allowed']);
                exit;
            }

            $deleted = $this->productModel->delete($id);
            if ($deleted) {
                echo json_encode(['success' => true]);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'Failed to delete product']);
            }
        });
    }
}
