<?php
session_start();
require_once __DIR__ . '/../bootstrap.php';

use App\Models\User;

if (!isset($_SESSION['user']) || $_SESSION['user']['role'] !== 'admin') {
    header('Location: login.php');
    exit;
}

$userModel = new User();
$errors = [];
$success = '';

// Handle form submission for new user
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $username = trim($_POST['username']);
    $password = $_POST['password'];
    $full_name = trim($_POST['full_name']);
    $email = trim($_POST['email']);
    $role = $_POST['role'];
    $branch_id = empty($_POST['branch_id']) ? null : $_POST['branch_id'];

    if (!$username || !$password || !$email || !$role) {
        $errors[] = "Username, Password, Email, and Role are required.";
    }

    // Check if username or email already exists
    $exists = $userModel->checkUserExists($username, $email);
    if ($exists > 0) {
        $errors[] = "User with this username or email already exists.";
    }

    if (empty($errors)) {
        $created = $userModel->createUser($username, $password, $full_name, $email, $role, $branch_id);
        if ($created) {
            $success = "User created successfully.";
        } else {
            $errors[] = "Failed to create new user.";
        }
    }
}

$users = $userModel->getAllUsers();
?>
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8" />
<title>Manage Users - Admin Panel</title>
<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet" />
</head>
<body>
<div class="container mt-4">
    <h1>Manage Users</h1>

    <?php if ($success): ?>
        <div class="alert alert-success"><?= htmlspecialchars($success) ?></div>
    <?php endif; ?>

    <?php if ($errors): ?>
        <div class="alert alert-danger">
            <ul>
                <?php foreach ($errors as $e): ?>
                    <li><?= htmlspecialchars($e) ?></li>
                <?php endforeach; ?>
            </ul>
        </div>
    <?php endif; ?>

    <h2>Add New User</h2>
    <form method="post" action="">
        <div class="mb-3">
            <label>Username:</label>
            <input type="text" name="username" class="form-control" required>
        </div>
        <div class="mb-3">
            <label>Password:</label>
            <input type="password" name="password" class="form-control" required>
        </div>
        <div class="mb-3">
            <label>Full Name:</label>
            <input type="text" name="full_name" class="form-control">
        </div>
        <div class="mb-3">
            <label>Email:</label>
            <input type="email" name="email" class="form-control" required>
        </div>
        <div class="mb-3">
            <label>Role:</label>
            <select name="role" class="form-select" required>
                <option value="admin">Admin</option>
                <option value="sales_rep">Sales Rep</option>
                <option value="cashier">Cashier</option>
                <option value="user">User</option>
            </select>
        </div>
        <div class="mb-3">
            <label>Branch ID (optional):</label>
            <input type="text" name="branch_id" class="form-control">
        </div>
        <button type="submit" class="btn btn-primary">Add User</button>
    </form>

    <hr>

    <h2>Existing Users</h2>
    <table class="table table-bordered">
        <thead class="table-dark">
            <tr>
                <th>User ID</th><th>Username</th><th>Email</th><th>Full Name</th><th>Role</th><th>Branch ID</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($users as $user): ?>
            <tr>
                <td><?= htmlspecialchars($user['user_id']) ?></td>
                <td><?= htmlspecialchars($user['username']) ?></td>
                <td><?= htmlspecialchars($user['email']) ?></td>
                <td><?= htmlspecialchars($user['full_name']) ?></td>
                <td><?= htmlspecialchars($user['role']) ?></td>
                <td><?= htmlspecialchars($user['branch_id']) ?></td>
            </tr>
            <th>Actions</th>
...
<td>
    <a href="edit_user.php?id=<?= $user['user_id'] ?>" class="btn btn-sm btn-warning">Edit</a>
    <a href="delete_user.php?id=<?= $user['user_id'] ?>" class="btn btn-sm btn-danger" onclick="return confirm('Delete this user?')">Delete</a>
</td>
            <?php endforeach; ?>
        </tbody>
    </table>
</div>
</body>
<nav>
  <a href="dashboard.php" class="btn btn-secondary">← Back to Dashboard</a>
</nav>
</html>
