<?php
require_once __DIR__ . '/../includes/database.php';
require_once __DIR__ . '/../config/config.php';

session_start();
require_once __DIR__ . '/../includes/role_helpers.php';
requireLogin();

$conn = getDbConnection();
$userRole = getUserRole();
$username = getUsername();

// Only allow access to admin users for this page
if ($userRole !== 'admin') {
    http_response_code(403);
    die('Access Denied');
}

// Get filter inputs with default values
$startDate = $_GET['start_date'] ?? date('Y-m-01');
$endDate = $_GET['end_date'] ?? date('Y-m-d');
$salesRepId = $_GET['sales_rep_id'] ?? '';

// Validate dates
if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $startDate)) $startDate = date('Y-m-01');
if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $endDate)) $endDate = date('Y-m-d');

// Fetch sales reps and cashiers for filter dropdown
try {
    $stmt = $conn->query("SELECT user_id, username, role FROM users WHERE role IN ('sales_rep', 'cashier') ORDER BY username");
    $staffList = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    error_log('Error fetching staff list: ' . $e->getMessage());
    $staffList = [];
}

// Build the main query with optional sales rep filter
$query = "
    SELECT 
        s.sale_id,
        s.sale_date,
        u.username AS sales_rep,
        c.customer_name,
        p.product_name,
        si.quantity,
        si.price,
        (si.quantity * si.price) AS total_price
    FROM sales s
    JOIN sale_items si ON s.sale_id = si.sale_id
    JOIN products p ON si.product_id = p.product_id
    JOIN users u ON s.sales_rep_id = u.user_id
    JOIN customers c ON s.customer_id = c.customer_id
    WHERE s.sale_date BETWEEN :start_date AND :end_date
";

$params = [
    ':start_date' => $startDate,
    ':end_date' => $endDate . ' 23:59:59',
];

if (!empty($salesRepId)) {
    $query .= " AND s.sales_rep_id = :sales_rep_id";
    $params[':sales_rep_id'] = $salesRepId;
}

$query .= " ORDER BY s.sale_date DESC LIMIT 100";

try {
    $stmt = $conn->prepare($query);
    $stmt->execute($params);
    $sales = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    error_log('Error fetching sales report: ' . $e->getMessage());
    die('Error loading sales report');
}

$conn = null;
?>

<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <title>Sales Report - Adwenpa Supermart</title>
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet"/>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/bootstrap-icons/1.13.0/font/bootstrap-icons.min.css" rel="stylesheet"/>
</head>

<body>

<nav class="navbar navbar-expand-lg navbar-dark bg-primary">
    <div class="container-fluid">
        <a class="navbar-brand" href="dashboard.php">Adwenpa Supermart</a>
        <div class="dropdown ms-auto">
            <a href="#" class="d-flex align-items-center text-white text-decoration-none dropdown-toggle" id="userMenu" data-bs-toggle="dropdown" aria-expanded="false">
                <i class="bi bi-person-circle fs-4 me-2"></i><?= htmlspecialchars($username) ?>
            </a>
            <ul class="dropdown-menu dropdown-menu-end" aria-labelledby="userMenu">
                <li><a class="dropdown-item" href="change_password.php">Change Password</a></li>
                <li><hr class="dropdown-divider"></li>
                <li><a class="dropdown-item" href="logout.php">Logout</a></li>
            </ul>
        </div>
    </div>
</nav>

<main class="container mt-4">

    <h1>Sales Report</h1>

    <form method="get" class="row g-3 mb-4">

        <div class="col-md-3">
            <label for="start_date" class="form-label">Start Date</label>
            <input type="date" id="start_date" name="start_date" value="<?= htmlspecialchars($startDate) ?>" class="form-control">
        </div>

        <div class="col-md-3">
            <label for="end_date" class="form-label">End Date</label>
            <input type="date" id="end_date" name="end_date" value="<?= htmlspecialchars($endDate) ?>" class="form-control">
        </div>

        <div class="col-md-4">
            <label for="sales_rep_id" class="form-label">Sales Rep / Cashier</label>
            <select name="sales_rep_id" id="sales_rep_id" class="form-select">
                <option value="">All</option>
                <?php foreach ($staffList as $staff): ?>
                    <option value="<?= $staff['user_id'] ?>" <?= ($salesRepId == $staff['user_id']) ? 'selected' : '' ?>>
                        <?= htmlspecialchars("{$staff['username']} ({$staff['role']})") ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>

        <div class="col-md-2 d-flex align-items-end">
            <button type="submit" class="btn btn-primary w-100">Filter</button>
        </div>

    </form>

    <?php if (count($sales) > 0): ?>
        <div class="table-responsive">
            <table class="table table-striped align-middle">
                <thead>
                <tr>
                    <th>Sale ID</th>
                    <th>Date</th>
                    <th>Sales Rep</th>
                    <th>Customer</th>
                    <th>Product</th>
                    <th>Qty</th>
                    <th>Price (GHS)</th>
                    <th>Total (GHS)</th>
                </tr>
                </thead>
                <tbody>
                <?php foreach ($sales as $row): ?>
                    <tr>
                        <td><?= htmlspecialchars($row['sale_id']) ?></td>
                        <td><?= htmlspecialchars(date('Y-m-d H:i', strtotime($row['sale_date']))) ?></td>
                        <td><?= htmlspecialchars($row['sales_rep']) ?></td>
                        <td><?= htmlspecialchars($row['customer_name']) ?></td>
                        <td><?= htmlspecialchars($row['product_name']) ?></td>
                        <td><?= htmlspecialchars($row['quantity']) ?></td>
                        <td><?= number_format($row['price'], 2) ?></td>
                        <td><?= number_format($row['total_price'], 2) ?></td>
                    </tr>
                <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    <?php else: ?>
        <p>No sales transactions found for the selected filters.</p>
    <?php endif; ?>

</main>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
