<?php
require_once __DIR__ . '/../includes/database.php';
require_once __DIR__ . '/../config/config.php';

session_start();
require_once __DIR__ . '/../includes/role_helpers.php';
requireLogin();

$conn = getDbConnection();
$userRole = getUserRole();
$username = getUsername();

// Sanitize and process input for report filters (e.g., date range)
$startDate = $_GET['start_date'] ?? date('Y-m-01'); // default start of current month
$endDate = $_GET['end_date'] ?? date('Y-m-d');      // default today

// Validate dates (optional)
if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $startDate)) $startDate = date('Y-m-01');
if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $endDate)) $endDate = date('Y-m-d');

try {
    // Example report: Total sales, grouped by day, within date range
    $stmt = $conn->prepare("
        SELECT DATE(sale_date) as sale_day, COUNT(*) AS total_sales, SUM(total_amount) AS total_revenue
        FROM sales
        WHERE sale_date BETWEEN :start AND :end
        GROUP BY sale_day
        ORDER BY sale_day DESC
        LIMIT 50
    ");
    $stmt->execute(['start' => $startDate, 'end' => $endDate]);
    $dailySales = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Inventory report: low stock items
    $lowStockThreshold = 10;
    $stmt2 = $conn->prepare("
        SELECT product_name, stock_quantity 
        FROM products
        WHERE stock_quantity < :threshold
        ORDER BY stock_quantity ASC
        LIMIT 20
    ");
    $stmt2->execute(['threshold' => $lowStockThreshold]);
    $lowStockItems = $stmt2->fetchAll(PDO::FETCH_ASSOC);

} catch (PDOException $e) {
    error_log('Report generation error: ' . $e->getMessage());
    die('Unable to generate reports at this time.');
}

$conn = null;
?>

<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8" />
    <title>Reports - Adwenpa Supermart</title>
    <meta name="viewport" content="width=device-width, initial-scale=1" />
    <link
        href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css"
        rel="stylesheet"
    />
    <link
        href="https://cdnjs.cloudflare.com/ajax/libs/bootstrap-icons/1.13.0/font/bootstrap-icons.min.css"
        rel="stylesheet"
    />
</head>

<body>
    <nav class="navbar navbar-expand-lg navbar-dark bg-primary">
        <div class="container-fluid">
            <a class="navbar-brand" href="dashboard.php">Adwenpa Supermart</a>
            <div class="d-flex">
                <div class="dropdown">
                    <a
                        href="#"
                        class="text-white text-decoration-none dropdown-toggle d-flex align-items-center"
                        id="userMenu"
                        data-bs-toggle="dropdown"
                        aria-expanded="false"
                    >
                        <i class="bi bi-person-circle fs-4 me-2"></i>
                        <?= htmlspecialchars($username) ?>
                    </a>
                    <ul
                        class="dropdown-menu dropdown-menu-end"
                        aria-labelledby="userMenu"
                    >
                        <li>
                            <a class="dropdown-item" href="change_password.php"
                                >Change Password</a
                            >
                        </li>
                        <li><hr class="dropdown-divider" /></li>
                        <li><a class="dropdown-item" href="logout.php">Logout</a></li>
                    </ul>
                </div>
            </div>
        </div>
    </nav>

    <main class="container mt-4">
        <h1>Reports</h1>

        <form method="get" class="row g-3 mb-4">
            <div class="col-md-5">
                <label for="start_date" class="form-label">Start Date</label>
                <input
                    type="date"
                    class="form-control"
                    id="start_date"
                    name="start_date"
                    value="<?= htmlspecialchars($startDate) ?>"
                />
            </div>
            <div class="col-md-5">
                <label for="end_date" class="form-label">End Date</label>
                <input
                    type="date"
                    class="form-control"
                    id="end_date"
                    name="end_date"
                    value="<?= htmlspecialchars($endDate) ?>"
                />
            </div>
            <div class="col-md-2 d-flex align-items-end">
                <button type="submit" class="btn btn-primary w-100">
                    Filter
                </button>
            </div>
        </form>

        <section>
            <h2>Daily Sales Report</h2>
            <?php if (count($dailySales) > 0) : ?>
            <table class="table table-striped">
                <thead>
                    <tr>
                        <th>Date</th>
                        <th>Total Sales</th>
                        <th>Total Revenue (GHS)</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($dailySales as $day) : ?>
                    <tr>
                        <td><?= htmlspecialchars($day['sale_day']) ?></td>
                        <td><?= htmlspecialchars($day['total_sales']) ?></td>
                        <td><?= number_format($day['total_revenue'], 2) ?></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            <?php else : ?>
            <p>No sales data available within the selected period.</p>
            <?php endif; ?>
        </section>

        <section class="mt-4">
            <h2>Low Stock Items (Below <?= $lowStockThreshold ?> units)</h2>
            <?php if (count($lowStockItems) > 0) : ?>
            <table class="table table-striped">
                <thead>
                    <tr>
                        <th>Product</th>
                        <th>Quantity</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($lowStockItems as $item) : ?>
                    <tr>
                        <td><?= htmlspecialchars($item['product_name']) ?></td>
                        <td><?= htmlspecialchars($item['stock_quantity']) ?></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            <?php else : ?>
            <p>All stock levels are sufficient.</p>
            <?php endif; ?>
        </section>
    </main>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
