<?php
// public/pos.php

session_start();

require_once __DIR__ . '/../bootstrap.php';
require_once __DIR__ . '/../vendor/autoload.php';

use App\Models\Product;
use App\Models\Unit;
use App\Models\Category;
use App\Models\Branch;
use App\Models\Sale;

if (!isset($_SESSION['user'])) {
    header('Location: login.php');
    exit;
}

$user = $_SESSION['user'];
$role = $user['role'] ?? '';

$productModel = new Product();
$unitModel = new Unit();
$categoryModel = new Category();
$branchModel = new Branch();
$saleModel = new Sale();

$products = $productModel->getAll();
$units = $unitModel->getAllUnits();
$categories = $categoryModel->getAllCategories();
$branches = $branchModel->getAllBranches();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8" />
    <title>POS - <?= htmlspecialchars($role) ?></title>
    <meta name="viewport" content="width=device-width, initial-scale=1" />
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet" />
    <style>
        #product-list, #cart { max-height: 500px; overflow-y: auto; }
        /* Print styling for receipt */
        @media print {
          body * { visibility: hidden; }
          #receipt, #receipt * { visibility: visible; }
          #receipt { position: absolute; left: 0; top: 0; width: 300px; font-family: monospace; }
        }
    </style>
</head>
<body>
<div class="container mt-3">
    <h1>Point of Sale (POS) - <?= htmlspecialchars(ucfirst($role)) ?></h1>

    <!-- Voice Search Section -->
    <div class="mb-3">
        <label for="productSearch" class="form-label">Search Products</label>
        <div class="input-group">
            <input type="text" id="productSearch" class="form-control" placeholder="Type or click microphone to search..." oninput="filterProducts()" />
            <button id="micButton" class="btn btn-outline-secondary" type="button" title="Search by voice">
                🎤
            </button>
        </div>
    </div>

    <div class="row">
        <div class="col-md-8">
            <h4>Products</h4>
            <div id="product-list" class="border p-2 mb-3">
                <table class="table table-sm table-hover">
                    <thead>
                        <tr>
                            <th>Name</th><th>Category</th><th>Price</th><th>Stock</th><th>Unit</th><th>Add</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($products as $p): ?>
                            <tr>
                                <td><?= htmlspecialchars($p['product_name']) ?></td>
                                <td><?= htmlspecialchars($p['category_name'] ?? '') ?></td>
                                <td><?= number_format($p['retail_price'], 2) ?></td>
                                <td><?= (int)$p['stock_quantity'] ?></td>
                                <td><?= htmlspecialchars($p['unit_name'] ?? '') ?></td>
                                <td>
                                    <button class="btn btn-sm btn-primary"
                                            onclick="addToCart(<?= $p['product_id'] ?>, '<?= addslashes($p['product_name']) ?>', <?= $p['retail_price'] ?>, '<?= addslashes($p['unit_name'] ?? '') ?>')">Add</button>
                                </td>
                            </tr>
                        <?php endforeach;?>
                    </tbody>
                </table>
            </div>
        </div>

        <div class="col-md-4">
            <h4>Cart</h4>
            <form id="pos-form" onsubmit="return processSale(event)">
                <div id="cart" class="border p-2 mb-3">
                    <table class="table table-sm">
                        <thead>
                            <tr>
                                <th>Name</th><th>Qty</th><th>Unit</th><th>Price</th><th>Total</th><th>Action</th>
                            </tr>
                        </thead>
                        <tbody id="cart-items">
                            <tr><td colspan="6" class="text-center">No items added</td></tr>
                        </tbody>
                    </table>
                </div>

                <div class="mb-3">
                    <label for="payment_method">Payment Method</label>
                    <select id="payment_method" name="payment_method" class="form-select">
                        <option value="cash">Cash</option>
                        <option value="card">Card</option>
                        <option value="mobile_money">Mobile Money</option>
                        <option value="insurance">Insurance</option>
                        <option value="other">Other</option>
                    </select>
                </div>

                <div class="mb-3">
                    <strong>Total Amount: GHS <span id="total_amount_display">0.00</span></strong>
                </div>

                <button type="submit" class="btn btn-success w-100">Checkout</button>
            </form>
            <div id="message" class="mt-2"></div>

            <!-- Hidden receipt for printing -->
            <div id="receipt" style="display:none; width:300px; font-family: monospace; border:1px solid #000; padding:10px;">
                <h3>Store Name</h3>
                <p>Address Line 1<br>Phone: 123-456-7890</p>
                <hr>
                <div id="receipt-items"></div>
                <hr>
                <p>Total: GHS <span id="receipt-total"></span></p>
                <p>Payment Method: <span id="receipt-payment"></span></p>
                <p>Thank you for your purchase!</p>
            </div>
        </div>
    </div>
</div>

<script>
    // Voice Recognition Search
    const micButton = document.getElementById('micButton');
    const productSearchInput = document.getElementById('productSearch');
    const SpeechRecognition = window.SpeechRecognition || window.webkitSpeechRecognition;

    if (!SpeechRecognition) {
        micButton.disabled = true;
        micButton.title = "Speech Recognition not supported by your browser";
    } else {
        const recognition = new SpeechRecognition();
        recognition.lang = 'en-US';
        recognition.interimResults = false;
        recognition.maxAlternatives = 1;

        micButton.addEventListener('click', () => {
            recognition.start();
            micButton.innerText = '🎙️... Listening';
        });

        recognition.addEventListener('result', (event) => {
            const transcript = event.results[0][0].transcript;
            productSearchInput.value = transcript;
            filterProducts();
            micButton.innerText = '🎤';
        });

        recognition.addEventListener('speechend', () => {
            recognition.stop();
            micButton.innerText = '🎤';
        });

        recognition.addEventListener('error', (event) => {
            alert('Speech recognition error: ' + event.error);
            micButton.innerText = '🎤';
        });
    }

    // Filter products in the table by search term
    function filterProducts() {
        const filter = productSearchInput.value.toLowerCase();
        const table = document.querySelector('#product-list table tbody');
        const trs = table.getElementsByTagName('tr');

        for (let tr of trs) {
            const text = tr.textContent.toLowerCase();
            tr.style.display = text.includes(filter) ? '' : 'none';
        }
    }

    // Store cart in memory and synchronize with IndexedDB
    let cart = [];

    // IndexedDB variables
    let db;
    const request = indexedDB.open('posDatabase', 1);

    request.onerror = (event) => {
        console.error('IndexedDB error:', event.target.error);
    };

    request.onupgradeneeded = (event) => {
        db = event.target.result;
        if (!db.objectStoreNames.contains('sales')) {
            const salesStore = db.createObjectStore('sales', { keyPath: 'sale_ref' });
            salesStore.createIndex('timestamp', 'timestamp', { unique: false });
        }
    };

    request.onsuccess = (event) => {
        db = event.target.result;
        console.log('IndexedDB initialized');
        loadOfflineSales();
    };

    // Save sale locally in IndexedDB
    function saveSaleOffline(saleData) {
        const transaction = db.transaction(['sales'], 'readwrite');
        const store = transaction.objectStore('sales');
        store.put(saleData);
        transaction.oncomplete = () => {
            console.log('Sale saved offline:', saleData);
        };
        transaction.onerror = (event) => {
            console.error('Error saving sale offline:', event.target.error);
        };
    }

    // Load offline sales for debugging or sync
    function loadOfflineSales() {
        const transaction = db.transaction(['sales'], 'readonly');
        const store = transaction.objectStore('sales');
        const request = store.getAll();
        request.onsuccess = () => {
            console.log('Offline sales loaded:', request.result);
        };
        request.onerror = (event) => {
            console.error('Error loading offline sales:', event.target.error);
        };
    }

    // Remove sale from offline store after sync
    function deleteOfflineSale(saleRef) {
        const transaction = db.transaction(['sales'], 'readwrite');
        const store = transaction.objectStore('sales');
        store.delete(saleRef);
        transaction.oncomplete = () => {
            console.log('Deleted synced sale:', saleRef);
        };
        transaction.onerror = (event) => {
            console.error('Error deleting sale:', event.target.error);
        };
    }

    // Detect offline/online status and attempt sync when online
    window.addEventListener('online', syncOfflineSales);

    async function syncOfflineSales() {
        if (!db) return;
        const transaction = db.transaction(['sales'], 'readonly');
        const store = transaction.objectStore('sales');
        const getAllRequest = store.getAll();

        getAllRequest.onsuccess = async () => {
            const offlineSales = getAllRequest.result;
            if (offlineSales.length === 0) return;

            try {
                const response = await fetch('sync-sales.php', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify(offlineSales)
                });
                const data = await response.json();
                if (data.success) {
                    // Remove synced sales
                    offlineSales.forEach(sale => deleteOfflineSale(sale.sale_ref));
                    alert('Offline sales synced successfully.');
                } else {
                    console.error('Sync failed:', data.error);
                }
            } catch (err) {
                console.error('Sync error:', err);
            }
        };
    }

    // Cart manipulation functions
    function addToCart(productId, productName, price, unit) {
        const existing = cart.find(i => i.product_id === productId);
        if (existing) {
            existing.quantity++;
        } else {
            cart.push({product_id: productId, product_name: productName, unit_price: price, unit: unit, quantity: 1});
        }
        renderCart();
    }

    function removeFromCart(productId) {
        cart = cart.filter(i => i.product_id !== productId);
        renderCart();
    }

    function updateQuantity(productId, qty) {
        const item = cart.find(i => i.product_id === productId);
        if (item) {
            const newQty = parseInt(qty);
            if (newQty > 0) {
                item.quantity = newQty;
            } else {
                removeFromCart(productId);
            }
            renderCart();
        }
    }

    function renderCart() {
        const tbody = document.getElementById('cart-items');
        tbody.innerHTML = '';
        if (cart.length === 0) {
            tbody.innerHTML = '<tr><td colspan="6" class="text-center">No items added</td></tr>';
            updateTotal(0);
            return;
        }
        let total = 0;
        cart.forEach(item => {
            const itemTotal = item.quantity * item.unit_price;
            total += itemTotal;
            const row = document.createElement('tr');
            row.innerHTML = `
                <td>${item.product_name}</td>
                <td><input type="number" min="1" value="${item.quantity}" onchange="updateQuantity(${item.product_id}, this.value)" class="form-control form-control-sm"></td>
                <td>${item.unit}</td>
                <td>${item.unit_price.toFixed(2)}</td>
                <td>${itemTotal.toFixed(2)}</td>
                <td><button type="button" onclick="removeFromCart(${item.product_id})" class="btn btn-sm btn-danger">X</button></td>
            `;
            tbody.appendChild(row);
        });
        updateTotal(total);
    }

    function updateTotal(amount) {
        document.getElementById('total_amount_display').textContent = amount.toFixed(2);
    }

    async function processSale(event) {
        event.preventDefault();
        const messageDiv = document.getElementById('message');
        if (cart.length === 0) {
            messageDiv.innerHTML = '<div class="alert alert-warning">Cart is empty.</div>';
            return false;
        }

        const paymentMethod = document.getElementById('payment_method').value;
        const saleData = {
            sale_ref: 'sale-' + Date.now(), // unique ref for offline sync
            total_amount: cart.reduce((s, i) => s + i.quantity * i.unit_price, 0),
            payment_method: paymentMethod,
            sales_rep_id: <?= $user['user_id'] ?>,
            branch_id: <?= $user['branch_id'] ?? 1 ?>,
            created_at: new Date().toISOString().slice(0, 19).replace('T', ' ')
        };

        const payload = {
            saleData: saleData,
            items: cart
        };

        if (navigator.onLine) {
            try {
                const response = await fetch('process_sale.php', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify(payload)
                });
                const data = await response.json();
                if (data.success) {
                    messageDiv.innerHTML = `<div class="alert alert-success">Sale #${data.sale_id} completed successfully.</div>`;
                    renderReceipt();
                    window.print();
                    cart = [];
                    renderCart();
                } else {
                    messageDiv.innerHTML = `<div class="alert alert-danger">${data.error || 'An error occurred'}</div>`;
                }
            } catch (error) {
                messageDiv.innerHTML = `<div class="alert alert-danger">Error: ${error.message}</div>`;
            }
        } else {
            // Save offline and show message
            saveSaleOffline({...saleData, items: cart});
            messageDiv.innerHTML = '<div class="alert alert-warning">You are currently offline. Sale saved locally and will sync when online.</div>';
            cart = [];
            renderCart();
        }
        return false;
    }

    function renderReceipt() {
        const receiptDiv = document.getElementById('receipt');
        const itemsDiv = document.getElementById('receipt-items');
        const totalSpan = document.getElementById('receipt-total');
        const paymentSpan = document.getElementById('receipt-payment');

        itemsDiv.innerHTML = '';
        cart.forEach(item => {
            itemsDiv.innerHTML += `<div>${item.product_name} x${item.quantity} @ GHS ${item.unit_price.toFixed(2)}</div>`;
        });
        const total = cart.reduce((sum, i) => sum + i.quantity * i.unit_price, 0);
        totalSpan.textContent = total.toFixed(2);
        paymentSpan.textContent = document.getElementById('payment_method').value;

        receiptDiv.style.display = 'block';
    }

    renderCart();
</script>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
