<?php
session_start();

require_once __DIR__ . '/../bootstrap.php';

use App\Models\Sale;

if (!isset($_SESSION['user'])) {
    header("Location: login.php");
    exit;
}

$user = $_SESSION['user'];
$saleModel = new Sale();

// Receive filter inputs from GET parameters
$startDate = $_GET['start_date'] ?? null;
$endDate = $_GET['end_date'] ?? null;

// Validate dates (basic format check)
if ($startDate && !preg_match('/^\d{4}-\d{2}-\d{2}$/', $startDate)) {
    $startDate = null;
}
if ($endDate && !preg_match('/^\d{4}-\d{2}-\d{2}$/', $endDate)) {
    $endDate = null;
}

// Fetch filtered sales for logged in sales rep
$sales = $saleModel->getSalesByRep($user['user_id'], $startDate, $endDate);

function esc($str) {
    return htmlspecialchars($str, ENT_QUOTES | ENT_HTML5, 'UTF-8');
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8" />
    <title>My Sales - Filtered</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet" />
</head>
<body>
<nav class="navbar navbar-expand-lg navbar-dark bg-dark">
    <a class="navbar-brand" href="dashboard.php">Adwenpa POS</a>
    <ul class="navbar-nav ms-auto">
        <li class="nav-item"><span class="nav-link text-white">Hello, <?= esc($user['username']) ?></span></li>
        <li class="nav-item"><a href="logout.php" class="nav-link">Logout</a></li>
    </ul>
</nav>

<div class="container mt-4">
    <h1>My Sales</h1>

    <!-- Filter Form -->
    <form method="get" class="row gy-2 gx-3 align-items-center mb-4">
        <div class="col-auto">
            <label for="start_date" class="form-label">Start Date</label>
            <input type="date" id="start_date" name="start_date" class="form-control" value="<?= esc($startDate) ?>">
        </div>
        <div class="col-auto">
            <label for="end_date" class="form-label">End Date</label>
            <input type="date" id="end_date" name="end_date" class="form-control" value="<?= esc($endDate) ?>">
        </div>
        <div class="col-auto pt-4">
            <button type="submit" class="btn btn-primary">Apply Filter</button>
            <a href="my_sales.php" class="btn btn-secondary">Reset</a>
        </div>
    </form>

    <?php if (empty($sales)): ?>
        <div class="alert alert-info">No sales found for the selected criteria.</div>
    <?php else: ?>
        <table class="table table-striped table-hover">
            <thead>
                <tr>
                    <th>Sale ID</th>
                    <th>Sale Date</th>
                    <th>Total Amount</th>
                    <th>Discount</th>
                    <th>Tax</th>
                    <th>Payment Method</th>
                    <th>Branch ID</th>
                    <th>Created At</th>
                    <th>Updated At</th>
                    <th>Details</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($sales as $sale): ?>
                <tr>
                    <td><?= esc($sale['sale_id']) ?></td>
                    <td><?= esc($sale['sale_date']) ?></td>
                    <td><?= esc(number_format($sale['total_amount'], 2)) ?></td>
                    <td><?= esc(number_format($sale['discount'], 2)) ?></td>
                    <td><?= esc(number_format($sale['tax'], 2)) ?></td>
                    <td><?= esc($sale['payment_method']) ?></td>
                    <td><?= esc($sale['branch_id']) ?></td>
                    <td><?= esc($sale['created_at']) ?></td>
                    <td><?= esc($sale['updated_at']) ?></td>
                    <td>
                        <a href="sale_details.php?sale_id=<?= esc($sale['sale_id']) ?>" class="btn btn-sm btn-primary">View Details</a>
                    </td>
                </tr>
                <?php endforeach;?>
            </tbody>
        </table>
    <?php endif; ?>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
