<?php
session_start();
require_once __DIR__ . '/../bootstrap.php';

use App\Models\StockMovement;
use App\Models\Inventory;
use App\Models\Branch;

if (!isset($_SESSION['user']) || $_SESSION['user']['role'] !== 'admin') {
    header("Location: login.php");
    exit;
}

$stockMovementModel = new StockMovement();
$inventoryModel = new Inventory();
$branchModel = new Branch();

$products = $inventoryModel->getAllItems();
$branches = $branchModel->getAllBranches();

$movement_id = $_GET['id'] ?? null;
if (!$movement_id || !is_numeric($movement_id)) {
    die("Invalid movement ID.");
}

$movement = $stockMovementModel->getMovementById((int)$movement_id);
if (!$movement) {
    die("Stock movement not found.");
}

$errors = [];
$success = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $product_id = (int)($_POST['product_id'] ?? 0);
    $branch_id = !empty($_POST['branch_id']) ? (int)$_POST['branch_id'] : null;
    $quantity = (int)($_POST['quantity'] ?? 0);
    $movement_type = $_POST['movement_type'] ?? '';
    $reference = trim($_POST['reference'] ?? '');

    if ($product_id <= 0) {
        $errors[] = "Please select a product.";
    }
    if ($quantity <= 0) {
        $errors[] = "Quantity must be greater than zero.";
    }
    if (!in_array($movement_type, ['in', 'out', 'adjustment'])) {
        $errors[] = "Invalid movement type selected.";
    }

    if (empty($errors)) {
        $updated = $stockMovementModel->updateMovement($movement_id, $product_id, $branch_id, $quantity, $movement_type, $reference);
        if ($updated) {
            $success = "Stock movement updated successfully.";
            $movement = $stockMovementModel->getMovementById((int)$movement_id);
        } else {
            $errors[] = "Failed to update stock movement.";
        }
    }
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8" />
    <title>Edit Stock Movement - Super Market POS</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet" />
</head>
<body>
<div class="container mt-4">
    <a href="stock_movements.php" class="btn btn-secondary mb-3">← Back to Movements</a>
    <h1>Edit Stock Movement</h1>

    <?php if ($success): ?>
        <div class="alert alert-success"><?= htmlspecialchars($success) ?></div>
    <?php endif; ?>

    <?php if ($errors): ?>
        <div class="alert alert-danger">
            <ul>
                <?php foreach ($errors as $e): ?><li><?= htmlspecialchars($e) ?></li><?php endforeach; ?>
            </ul>
        </div>
    <?php endif; ?>

    <form method="post" action="">
        <div class="mb-3">
            <label for="product_id" class="form-label">Product *</label>
            <select name="product_id" id="product_id" class="form-select" required>
                <option value="">Select product</option>
                <?php foreach ($products as $p): ?>
                <option value="<?= $p['inventory_id'] ?>" <?= ($movement['product_id'] == $p['inventory_id']) ? 'selected' : '' ?>>
                    <?= htmlspecialchars($p['product_name']) ?>
                </option>
                <?php endforeach; ?>
            </select>
        </div>
        <div class="mb-3">
            <label for="branch_id" class="form-label">Branch (optional)</label>
            <select name="branch_id" id="branch_id" class="form-select">
                <option value="">Select branch (optional)</option>
                <?php foreach ($branches as $b): ?>
                <option value="<?= $b['branch_id'] ?>" <?= ($movement['branch_id'] == $b['branch_id']) ? 'selected' : '' ?>>
                    <?= htmlspecialchars($b['branch_name']) ?>
                </option>
                <?php endforeach; ?>
            </select>
        </div>
        <div class="mb-3">
            <label for="quantity" class="form-label">Quantity *</label>
            <input type="number" min="1" name="quantity" id="quantity" class="form-control" required value="<?= (int)$movement['quantity'] ?>">
        </div>
        <div class="mb-3">
            <label for="movement_type" class="form-label">Movement Type *</label>
            <select name="movement_type" id="movement_type" class="form-select" required>
                <option value="in" <?= ($movement['movement_type'] === 'in') ? 'selected' : '' ?>>In</option>
                <option value="out" <?= ($movement['movement_type'] === 'out') ? 'selected' : '' ?>>Out</option>
                <option value="adjustment" <?= ($movement['movement_type'] === 'adjustment') ? 'selected' : '' ?>>Adjustment</option>
            </select>
        </div>
        <div class="mb-3">
            <label for="reference" class="form-label">Reference</label>
            <input type="text" name="reference" id="reference" class="form-control" value="<?= htmlspecialchars($movement['reference'] ?? '') ?>">
        </div>
        <button type="submit" class="btn btn-primary">Update Movement</button>
    </form>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
