<?php
session_start();
require_once __DIR__ . '/../bootstrap.php';

use App\Models\Inventory;
use App\Models\Branch;

if (!isset($_SESSION['user']) || $_SESSION['user']['role'] !== 'admin') {
    header("Location: login.php");
    exit;
}

$inventoryModel = new Inventory();
$branchModel = new Branch();
$errors = [];
$success = '';

// Get inventory item ID from query string
$inventory_id = $_GET['id'] ?? null;
if (!$inventory_id || !is_numeric($inventory_id)) {
    die("Invalid inventory ID.");
}

// Fetch existing item data
$item = $inventoryModel->getItemById((int)$inventory_id);
if (!$item) {
    die("Inventory item not found.");
}

// Fetch branches for dropdown
$branches = $branchModel->getAllBranches();

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $product_name = trim($_POST['product_name']);
    $sku = trim($_POST['sku']);
    $quantity = (int) $_POST['quantity'];
    $purchase_price = (float) $_POST['purchase_price'];
    $selling_price = (float) $_POST['selling_price'];
    $branch_id = !empty($_POST['branch_id']) ? (int) $_POST['branch_id'] : null;

    // Validations
    if (!$product_name) {
        $errors[] = "Product name is required.";
    }
    if (!$sku) {
        $errors[] = "SKU/Barcode is required.";
    }
    if ($quantity < 0) {
        $errors[] = "Quantity cannot be negative.";
    }
    if ($purchase_price <= 0) {
        $errors[] = "Purchase price must be greater than zero.";
    }
    if ($selling_price <= 0) {
        $errors[] = "Selling price must be greater than zero.";
    }

    if (empty($errors)) {
        $updated = $inventoryModel->updateItem($inventory_id, $product_name, $sku, $quantity, $purchase_price, $selling_price, $branch_id);
        if ($updated) {
            $success = "Inventory item updated successfully.";
            $item = $inventoryModel->getItemById((int)$inventory_id); // Refresh data after update
        } else {
            $errors[] = "Failed to update inventory item.";
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8" />
    <title>Edit Inventory Item - Super Market POS</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet"/>
</head>
<body>
<div class="container mt-4">
    <a href="inventory.php" class="btn btn-secondary mb-3">← Back to Inventory</a>
    <h1>Edit Inventory Item</h1>

    <?php if ($success): ?>
        <div class="alert alert-success"><?= htmlspecialchars($success) ?></div>
    <?php endif; ?>

    <?php if ($errors): ?>
        <div class="alert alert-danger">
            <ul>
            <?php foreach ($errors as $e): ?>
                <li><?= htmlspecialchars($e) ?></li>
            <?php endforeach; ?>
            </ul>
        </div>
    <?php endif; ?>

    <form method="post" action="">
        <div class="mb-3">
            <label for="product_name" class="form-label">Product Name *</label>
            <input type="text" id="product_name" name="product_name" class="form-control" required value="<?= htmlspecialchars($item['product_name']) ?>">
        </div>
        <div class="mb-3">
            <label for="sku" class="form-label">SKU / Barcode *</label>
            <input type="text" id="sku" name="sku" class="form-control" required value="<?= htmlspecialchars($item['sku']) ?>">
        </div>
        <div class="mb-3">
            <label for="quantity" class="form-label">Quantity *</label>
            <input type="number" id="quantity" name="quantity" class="form-control" required min="0" value="<?= (int)$item['quantity'] ?>">
        </div>
        <div class="mb-3">
            <label for="purchase_price" class="form-label">Purchase Price (per unit) *</label>
            <input type="number" id="purchase_price" name="purchase_price" class="form-control" required step="0.01" min="0.01" value="<?= number_format($item['purchase_price'], 2, '.', '') ?>">
        </div>
        <div class="mb-3">
            <label for="selling_price" class="form-label">Selling Price (per unit) *</label>
            <input type="number" id="selling_price" name="selling_price" class="form-control" required step="0.01" min="0.01" value="<?= number_format($item['selling_price'], 2, '.', '') ?>">
        </div>
        <div class="mb-3">
            <label for="branch_id" class="form-label">Branch (optional)</label>
            <select id="branch_id" name="branch_id" class="form-select">
                <option value="">Select branch (optional)</option>
                <?php foreach ($branches as $branch): ?>
                    <option value="<?= $branch['branch_id'] ?>" <?= ($item['branch_id'] == $branch['branch_id']) ? 'selected' : '' ?>>
                        <?= htmlspecialchars($branch['branch_name']) ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>

        <button type="submit" class="btn btn-primary">Update Item</button>
    </form>
</div>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
