<?php
require_once __DIR__ . '/../includes/database.php';
require_once __DIR__ . '/../config/config.php';

session_start();
require_once __DIR__ . '/../includes/role_helpers.php';
requireLogin();

$conn = getDbConnection();

function fetchCount(PDO $conn, string $query, array $params = []): int {
    try {
        $stmt = $conn->prepare($query);
        $stmt->execute($params);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return $result ? (int)$result['total'] : 0;
    } catch (PDOException $e) {
        error_log("Query error ($query): " . $e->getMessage());
        return 0;
    }
}

// Fetch core counts
$total_products = fetchCount($conn, "SELECT COUNT(*) AS total FROM products");
$total_customers = fetchCount($conn, "SELECT COUNT(*) AS total FROM customers");
$total_suppliers = fetchCount($conn, "SELECT COUNT(*) AS total FROM suppliers");
$total_branches = fetchCount($conn, "SELECT COUNT(*) AS total FROM branches");
$low_stock_threshold = 10;
$low_stock_items = fetchCount($conn, "SELECT COUNT(*) AS total FROM products WHERE stock_quantity < :threshold", ['threshold' => $low_stock_threshold]);
$total_stock = fetchCount($conn, "SELECT COUNT(*) AS total FROM products WHERE stock_quantity > 0");

// Role info
$userRole = getUserRole();
$username = getUsername();

// Sales summary for admins
$total_sales_amount = 0;
if ($userRole === 'admin' || $userRole === 'manager') {
    try {
        $stmt = $conn->query("SELECT IFNULL(SUM(total_amount), 0) AS sales_sum FROM sales");
        $total_sales_amount = $stmt ? (float)$stmt->fetch(PDO::FETCH_ASSOC)['sales_sum'] : 0;
    } catch (PDOException $e) {
        error_log("Failed to get total sales amount: " . $e->getMessage());
    }
}

$conn = null;
?>

<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8" />
    <title>Dashboard - Adwenpa Supermart</title>
    <meta name="viewport" content="width=device-width, initial-scale=1, maximum-scale=1" />
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet" />
    <link href="https://cdnjs.cloudflare.com/ajax/libs/bootstrap-icons/1.13.0/font/bootstrap-icons.min.css" rel="stylesheet" />
    <style>
        body {
            min-height: 100vh;
            display: flex;
            flex-direction: column;
        }
        .sidebar {
            height: 100vh;
            position: fixed;
            top: 0;
            left: 0;
            width: 220px;
            background-color: #0d6efd;
            color: white;
            padding-top: 60px;
            flex-shrink: 0;
            z-index: 1040;
            overflow-y: auto;
            scrollbar-width: thin;
            scrollbar-color: #084298 transparent;
            -webkit-overflow-scrolling: touch;
        }
        .sidebar::-webkit-scrollbar {
            width: 8px;
        }
        .sidebar::-webkit-scrollbar-track {
            background: transparent;
        }
        .sidebar::-webkit-scrollbar-thumb {
            background-color: #084298;
            border-radius: 4px;
            border: 2px solid transparent;
            background-clip: content-box;
        }
        .sidebar a {
            color: white;
            text-decoration: none;
            padding: 12px 20px;
            display: flex;
            align-items: center;
            transition: background-color 0.2s, color 0.2s;
            position: relative;
            white-space: nowrap;
        }
        .sidebar a:hover,
        .sidebar a.active {
            background-color: #084298;
            color: white;
        }
        .sidebar a i {
            margin-right: 10px;
            font-size: 1.25rem;
        }
        main.content {
            margin-left: 220px;
            padding: 30px 20px;
            margin-top: 70px;
            flex-grow: 1;
        }
        .counter {
            font-weight: 700;
            font-size: 2.5rem;
        }
        #connection-status {
            position: fixed;
            bottom: 10px;
            right: 10px;
            padding: 10px 15px;
            background: #ffc107;
            color: #212529;
            border-radius: 5px;
            display: none;
            font-weight: 600;
            z-index: 1100;
        }
        @media (max-width: 991.98px) {
            .sidebar {
                width: 100%;
                height: auto;
                position: relative;
                padding-top: 0;
            }
            main.content {
                margin-left: 0;
                margin-top: 80px;
            }
        }
    </style>
</head>

<body>

    <nav class="navbar navbar-expand-lg navbar-dark bg-primary fixed-top">
        <div class="container-fluid">
            <a class="navbar-brand fw-bold" href="#">Adwenpa Supermart</a>
            <div class="dropdown ms-auto">
                <a href="#" class="d-flex align-items-center text-white text-decoration-none dropdown-toggle"
                    id="userMenu" data-bs-toggle="dropdown" aria-expanded="false">
                    <i class="bi bi-person-circle fs-4 me-2"></i><?= htmlspecialchars($username) ?>
                </a>
                <ul class="dropdown-menu dropdown-menu-end" aria-labelledby="userMenu">
                    <li><a class="dropdown-item" href="change_password.php">Change Password</a></li>
                    <li><hr class="dropdown-divider"></li>
                    <li><a class="dropdown-item" href="logout.php">Logout</a></li>
                </ul>
            </div>
        </div>
    </nav>

    <div class="sidebar d-none d-lg-block">
        <a href="dashboard.php" class="active"><i class="bi bi-speedometer2"></i> Dashboard</a>
        <a href="products.php"><i class="bi bi-box-seam"></i> Products</a>
        <?php if ($userRole === 'admin' || $userRole === 'manager'): ?>
        <a href="orders.php"><i class="bi bi-bag-check"></i> Orders</a>
        <a href="branches.php"><i class="bi bi-diagram-3"></i> Branches</a>
        <a href="suppliers.php"><i class="bi bi-truck"></i> Suppliers</a>
        <?php endif; ?>
        <?php if ($userRole === 'admin'): ?>
        <a href="users.php"><i class="bi bi-person-badge"></i> User Management</a>
        <a href="sales_report.php"><i class="bi bi-file-earmark-bar-graph"></i> Sales Report</a>
        <?php endif; ?>

        <a href="pos.php"><i class="bi bi-cash-stack"></i> POS</a>
        <a href="audit_logs.php"><i class="bi bi-file-earmark-text"></i> Audit Logs</a>
        <a href="inventory.php"><i class="bi bi-briefcase"></i> Inventory</a>
        <a href="stock.php"><i class="bi bi-inboxes"></i> Stock</a>
        <a href="backup.php"><i class="bi bi-hdd-network"></i> Backup</a>
        <a href="restore.php"><i class="bi bi-hdd-stack"></i> Restore</a>
        <a href="cashier_portal.php"><i class="bi bi-cash-coin"></i> Cashier Portal</a>
        <a href="my_sales.php"><i class="bi bi-bar-chart"></i> My Sales</a>
        <a href="reports.php"><i class="bi bi-graph-up"></i> Reports</a>
    </div>

    <main class="content">

        <div class="welcome mb-4">
            <h1>Welcome, <?= htmlspecialchars($username) ?>!</h1>
            <p>You are logged in as <strong><?= htmlspecialchars(ucfirst($userRole)) ?></strong>.</p>
        </div>

        <div class="row mb-4">

            <div class="col-md-3 mb-3">
                <div class="card shadow-sm border-0 text-center">
                    <div class="card-body">
                        <div class="counter text-primary" data-target="<?= $total_products ?>">0</div>
                        <div>Total Products</div>
                    </div>
                </div>
            </div>

            <div class="col-md-3 mb-3">
                <div class="card shadow-sm border-0 text-center">
                    <div class="card-body">
                        <div class="counter text-success" data-target="<?= $total_customers ?>">0</div>
                        <div>Total Customers</div>
                    </div>
                </div>
            </div>

            <div class="col-md-3 mb-3">
                <div class="card shadow-sm border-0 text-center">
                    <div class="card-body">
                        <div class="counter text-warning" data-target="<?= $total_stock ?>">0</div>
                        <div>Available Stock</div>
                    </div>
                </div>
            </div>

            <div class="col-md-3 mb-3">
                <div class="card shadow-sm border-0 text-center">
                    <div class="card-body">
                        <div class="counter text-danger" data-target="<?= $low_stock_items ?>">0</div>
                        <div>Low Stock Items</div>
                    </div>
                </div>
            </div>

            <?php if ($userRole === 'admin'): ?>
            <div class="col-md-3 mb-3">
                <div class="card shadow-sm border-0 text-center">
                    <div class="card-body">
                        <div class="counter text-info" data-target="<?= number_format($total_sales_amount, 2) ?>">0</div>
                        <div>Total Sales Amount (GHS)</div>
                    </div>
                </div>
            </div>
            <?php endif; ?>

        </div>

        <div id="connection-status">You are currently offline</div>
    </main>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>

    <script>
        // Animate counters
        document.addEventListener('DOMContentLoaded', () => {
            document.querySelectorAll('.counter').forEach(counter => {
                const updateCount = () => {
                    const target = +counter.getAttribute('data-target');
                    let count = +counter.innerText;
                    const increment = Math.ceil(target / 100);
                    if (count < target) {
                        counter.innerText = Math.min(count + increment, target);
                        setTimeout(updateCount, 25);
                    } else {
                        counter.innerText = target;
                    }
                };
                updateCount();
            });
        });

        // Online/offline status indicator
        function updateOnlineStatus() {
            const status = document.getElementById('connection-status');
            if (navigator.onLine) {
                status.style.display = "none";
            } else {
                status.style.display = "block";
            }
        }
        window.addEventListener('online', updateOnlineStatus);
        window.addEventListener('offline', updateOnlineStatus);
        updateOnlineStatus();
    </script>

    <!-- Service worker registration -->
    <script>
      if ('serviceWorker' in navigator) {
        window.addEventListener('load', function() {
          navigator.serviceWorker.register('/service-worker.js').then(
            function(registration) { console.log('ServiceWorker registered:', registration.scope); },
            function(err) { console.log('ServiceWorker registration failed:', err); }
          );
        });
      }
    </script>

</body>
</html>
