<?php
// public/create_po.php

session_start();

require_once __DIR__ . '/../bootstrap.php';
require_once __DIR__ . '/../vendor/autoload.php';

use App\Models\Product;
use App\Models\Supplier;
use App\Models\Branch;
use App\Models\PurchaseOrder;

if (!isset($_SESSION['user'])) {
    header('Location: login.php');
    exit;
}

$productModel = new Product();
$supplierModel = new Supplier();
$branchModel = new Branch();
$purchaseOrderModel = new PurchaseOrder();

$products = $productModel->getAll();
$suppliers = $supplierModel->getAllSuppliers();
$branches = $branchModel->getAllBranches();

$errors = [];
$success = '';
$supplierId = $branchId = null;
$orderDate = date('Y-m-d');

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $supplierId = (int)($_POST['supplier_id'] ?? 0);
    $branchId = (int)($_POST['branch_id'] ?? 0);
    $orderDate = $_POST['order_date'] ?? date('Y-m-d');
    $items = $_POST['items'] ?? [];

    // Validate inputs
    if ($supplierId <= 0) {
        $errors[] = "Please select a valid supplier.";
    }
    if ($branchId <= 0) {
        $errors[] = "Please select a valid branch.";
    }
    if (empty($items) || !is_array($items)) {
        $errors[] = "Please add at least one product.";
    }

    // Validate items
    $validItems = [];
    foreach ($items as $item) {
        $productId = (int)($item['product_id'] ?? 0);
        $quantity = (int)($item['quantity'] ?? 0);
        $unitPrice = floatval($item['unit_price'] ?? 0);

        if ($productId > 0 && $quantity > 0 && $unitPrice > 0) {
            $validItems[] = [
                'product_id' => $productId,
                'quantity' => $quantity,
                'unit_price' => $unitPrice
            ];
        }
    }

    if (empty($validItems)) {
        $errors[] = "Please add at least one product with valid quantity and price.";
    }

    // If no errors, save purchase order
    if (empty($errors)) {
        try {
            $poId = $purchaseOrderModel->createPurchaseOrder($supplierId, $branchId, $orderDate, $validItems);
            $success = "Purchase Order #$poId created successfully.";
            // Clear form values after successful insert
            $supplierId = $branchId = null;
            $orderDate = date('Y-m-d');
            $validItems = [];
        } catch (Exception $e) {
            $errors[] = "Failed to create purchase order: " . $e->getMessage();
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8" />
    <title>Create Purchase Order</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet" />
    <style>
        .remove-btn {
            cursor: pointer;
            color: red;
            font-weight: bold;
        }
    </style>
</head>
<body>
<div class="container mt-4">
    <h1>Create Purchase Order</h1>

    <?php if ($success): ?>
        <div class="alert alert-success"><?= htmlspecialchars($success) ?></div>
    <?php endif; ?>
    <?php if ($errors): ?>
        <div class="alert alert-danger">
            <ul>
                <?php foreach ($errors as $err): ?>
                    <li><?= htmlspecialchars($err) ?></li>
                <?php endforeach; ?>
            </ul>
        </div>
    <?php endif; ?>

    <form method="POST" id="poForm">
        <div class="mb-3">
            <label for="supplier_id" class="form-label">Supplier</label>
            <select id="supplier_id" name="supplier_id" class="form-select" required>
                <option value="">-- Select Supplier --</option>
                <?php foreach ($suppliers as $supplier): ?>
                    <option value="<?= $supplier['supplier_id'] ?>" <?= ($supplierId == $supplier['supplier_id']) ? 'selected' : '' ?>>
                        <?= htmlspecialchars($supplier['name']) ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>

        <div class="mb-3">
            <label for="branch_id" class="form-label">Branch</label>
            <select id="branch_id" name="branch_id" class="form-select" required>
                <option value="">-- Select Branch --</option>
                <?php foreach ($branches as $branch): ?>
                    <option value="<?= $branch['branch_id'] ?>" <?= ($branchId == $branch['branch_id']) ? 'selected' : '' ?>>
                        <?= htmlspecialchars($branch['branch_name']) ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>

        <div class="mb-3">
            <label for="order_date" class="form-label">Order Date</label>
            <input type="date" id="order_date" name="order_date" class="form-control" value="<?= htmlspecialchars($orderDate) ?>" required>
        </div>

        <h4>Products</h4>
        <table class="table table-bordered" id="productsTable">
            <thead>
                <tr>
                    <th>Product</th>
                    <th>Quantity</th>
                    <th>Unit Price (GHS)</th>
                    <th>Action</th>
                </tr>
            </thead>
            <tbody>
            <?php if (!empty($validItems)): ?>
                <?php foreach ($validItems as $item): ?>
                    <tr>
                        <td>
                            <select name="items[][product_id]" class="form-select" required>
                                <option value="">-- Select Product --</option>
                                <?php foreach ($products as $product): ?>
                                    <option value="<?= $product['product_id'] ?>" <?= ($product['product_id'] == $item['product_id']) ? 'selected' : '' ?>>
                                        <?= htmlspecialchars($product['product_name']) ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </td>
                        <td>
                            <input type="number" name="items[][quantity]" class="form-control" min="1" value="<?= $item['quantity'] ?>" required>
                        </td>
                        <td>
                            <input type="number" step="0.01" name="items[][unit_price]" class="form-control" min="0" value="<?= $item['unit_price'] ?>" required>
                        </td>
                        <td><span class="remove-btn" onclick="removeProductRow(this)">×</span></td>
                    </tr>
                <?php endforeach; ?>
            <?php else: ?>
                <tr>
                    <td>
                        <select name="items[][product_id]" class="form-select" required>
                            <option value="">-- Select Product --</option>
                            <?php foreach ($products as $product): ?>
                                <option value="<?= $product['product_id'] ?>">
                                    <?= htmlspecialchars($product['product_name']) ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </td>
                    <td><input type="number" name="items[][quantity]" class="form-control" min="1" value="1" required></td>
                    <td><input type="number" step="0.01" name="items[][unit_price]" class="form-control" min="0" value="0" required></td>
                    <td><span class="remove-btn" onclick="removeProductRow(this)">×</span></td>
                </tr>
            <?php endif; ?>
            </tbody>
        </table>

        <button type="button" class="btn btn-secondary mb-3" onclick="addProductRow()">Add Product</button>
        <button type="submit" class="btn btn-primary">Create Purchase Order</button>
    </form>
</div>

<script>
    let products = <?= json_encode($products) ?>;
    function addProductRow() {
        const tbody = document.querySelector('#productsTable tbody');
        const row = document.createElement('tr');
        const productOptions = products.map(p => `<option value="${p.product_id}">${p.product_name}</option>`).join('');
        row.innerHTML = `
            <td>
                <select name="items[][product_id]" class="form-select" required>
                    <option value="">-- Select Product --</option>
                    ${productOptions}
                </select>
            </td>
            <td><input type="number" name="items[][quantity]" class="form-control" min="1" value="1" required></td>
            <td><input type="number" step="0.01" name="items[][unit_price]" class="form-control" min="0" value="0" required></td>
            <td><span class="remove-btn" onclick="removeProductRow(this)">×</span></td>
        `;
        tbody.appendChild(row);
    }
    function removeProductRow(el) {
        const row = el.closest('tr');
        row.remove();
    }
</script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
