<?php
// public/admin_audit_logs.php

session_start();

require_once __DIR__ . '/../bootstrap.php';
require_once __DIR__ . '/../vendor/autoload.php';

use App\Models\AuditLog;

// Admin check
if (!isset($_SESSION['user']) || $_SESSION['user']['role'] !== 'admin') {
    header('Location: login.php');
    exit;
}

$auditLogModel = new AuditLog();

// Pagination settings
$limit = 20;
$page = isset($_GET['page']) && is_numeric($_GET['page']) ? (int)$_GET['page'] : 1;
$offset = ($page - 1) * $limit;

// Filtering parameters
$filterUser = $_GET['user'] ?? '';
$filterAction = $_GET['action'] ?? '';

// Build WHERE clause for filters
$where = [];
$params = [];
if ($filterUser !== '') {
    $where[] = 'u.username LIKE :user';
    $params[':user'] = '%' . $filterUser . '%';
}
if ($filterAction !== '') {
    $where[] = 'al.action LIKE :action';
    $params[':action'] = '%' . $filterAction . '%';
}
$whereSql = count($where) > 0 ? 'WHERE ' . implode(' AND ', $where) : '';

// Use getConnection() method to access PDO connection
$stmtCount = $auditLogModel->getConnection()->prepare("SELECT COUNT(*) FROM audit_logs al JOIN users u ON al.user_id = u.user_id $whereSql");
$stmtCount->execute($params);
$totalRows = (int)$stmtCount->fetchColumn();
$totalPages = ceil($totalRows / $limit);

// Get filtered logs with limit and offset
$sql = "SELECT al.*, u.username FROM audit_logs al JOIN users u ON al.user_id = u.user_id
        $whereSql ORDER BY al.created_at DESC LIMIT :limit OFFSET :offset";
$stmt = $auditLogModel->getConnection()->prepare($sql);

foreach ($params as $key => &$val) {
    $stmt->bindParam($key, $val, PDO::PARAM_STR);
}

$stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
$stmt->bindParam(':offset', $offset, PDO::PARAM_INT);
$stmt->execute();
$logs = $stmt->fetchAll(PDO::FETCH_ASSOC);

function queryString(array $overrides = []) {
    $params = $_GET;
    foreach ($overrides as $k => $v) {
        if ($v === null) {
            unset($params[$k]);
        } else {
            $params[$k] = $v;
        }
    }
    return http_build_query($params);
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8" />
    <title>Admin Audit Logs</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet" />
</head>
<body>
<div class="container mt-4">
    <h1>Audit Logs</h1>

    <form class="row gy-2 gx-3 align-items-center mb-3" method="get" action="">
        <div class="col-auto">
            <input type="text" class="form-control" name="user" placeholder="Filter by user" value="<?= htmlspecialchars($filterUser) ?>">
        </div>
        <div class="col-auto">
            <input type="text" class="form-control" name="action" placeholder="Filter by action" value="<?= htmlspecialchars($filterAction) ?>">
        </div>
        <div class="col-auto">
            <button class="btn btn-primary" type="submit">Filter</button>
            <a href="?<?= queryString(['user' => null, 'action' => null, 'page' => null]) ?>" class="btn btn-secondary">Clear</a>
        </div>
    </form>

    <table class="table table-bordered table-striped">
        <thead>
            <tr>
                <th>Log ID</th>
                <th>User</th>
                <th>Action</th>
                <th>Details</th>
                <th>Date</th>
            </tr>
        </thead>
        <tbody>
        <?php if (empty($logs)): ?>
            <tr><td colspan="5" class="text-center">No logs found.</td></tr>
        <?php else: ?>
            <?php foreach ($logs as $log): ?>
                <tr>
                    <td><?= htmlspecialchars($log['log_id']) ?></td>
                    <td><?= htmlspecialchars($log['username'] ?? 'Unknown') ?></td>
                    <td><?= htmlspecialchars($log['action']) ?></td>
                    <td>
                      <details>
                        <summary>View Details</summary>
                        <pre><?= htmlspecialchars($log['details']) ?></pre>
                      </details>
                    </td>
                    <td><?= htmlspecialchars($log['created_at']) ?></td>
                </tr>
            <?php endforeach; ?>
        <?php endif; ?>
        </tbody>
    </table>

    <!-- Pagination -->
    <nav>
        <ul class="pagination">
            <li class="page-item <?= $page <= 1 ? 'disabled' : '' ?>">
                <a class="page-link" href="?<?= queryString(['page' => $page - 1 > 1 ? $page - 1 : 1]) ?>">Previous</a>
            </li>
            <?php for ($p=1; $p <= $totalPages; $p++): ?>
                <li class="page-item <?= $page === $p ? 'active' : '' ?>">
                    <a class="page-link" href="?<?= queryString(['page' => $p]) ?>"><?= $p ?></a>
                </li>
            <?php endfor; ?>
            <li class="page-item <?= $page >= $totalPages ? 'disabled' : '' ?>">
                <a class="page-link" href="?<?= queryString(['page' => $page + 1 <= $totalPages ? $page + 1 : $totalPages]) ?>">Next</a>
            </li>
        </ul>
    </nav>

    <a href="dashboard.php" class="btn btn-secondary">Back to Dashboard</a>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
