<?php
session_start();
require_once __DIR__ . '/../bootstrap.php';

use App\Models\Inventory;
use App\Models\Branch;

if (!isset($_SESSION['user']) || $_SESSION['user']['role'] !== 'admin') {
    header("Location: login.php");
    exit;
}

$inventoryModel = new Inventory();
$branchModel = new Branch();
$errors = [];
$success = '';

// Fetch branches for dropdown selection
$branches = $branchModel->getAllBranches();

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $product_name = trim($_POST['product_name']);
    $sku = trim($_POST['sku']);
    $quantity = (int) $_POST['quantity'];
    $purchase_price = (float) $_POST['purchase_price'];
    $selling_price = (float) $_POST['selling_price'];
    $branch_id = !empty($_POST['branch_id']) ? (int) $_POST['branch_id'] : null;

    // Basic validations
    if (!$product_name) {
        $errors[] = "Product name is required.";
    }
    if (!$sku) {
        $errors[] = "SKU/Barcode is required.";
    }
    if ($quantity < 0) {
        $errors[] = "Quantity cannot be negative.";
    }
    if ($purchase_price <= 0) {
        $errors[] = "Purchase price must be greater than zero.";
    }
    if ($selling_price <= 0) {
        $errors[] = "Selling price must be greater than zero.";
    }

    // If no errors, create inventory item
    if (empty($errors)) {
        $created = $inventoryModel->createItem($product_name, $sku, $quantity, $purchase_price, $selling_price, $branch_id);

        if ($created) {
            $success = "Inventory item added successfully.";
            // Optionally clear form or redirect to inventory.php
        } else {
            $errors[] = "Failed to add inventory item. SKU might already exist.";
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8" />
    <title>Add Inventory Item - Super Market POS</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet" />
</head>
<body>
<div class="container mt-4">
    <a href="inventory.php" class="btn btn-secondary mb-3">← Back to Inventory</a>
    <h1>Add New Inventory Item</h1>

    <?php if ($success): ?>
        <div class="alert alert-success"><?= htmlspecialchars($success) ?></div>
    <?php endif; ?>

    <?php if ($errors): ?>
        <div class="alert alert-danger">
            <ul>
            <?php foreach ($errors as $e): ?>
                <li><?= htmlspecialchars($e) ?></li>
            <?php endforeach; ?>
            </ul>
        </div>
    <?php endif; ?>

    <form method="post" action="">
        <div class="mb-3">
            <label for="product_name" class="form-label">Product Name *</label>
            <input type="text" name="product_name" id="product_name" class="form-control" required value="<?= isset($product_name) ? htmlspecialchars($product_name) : '' ?>">
        </div>

        <div class="mb-3">
            <label for="sku" class="form-label">SKU / Barcode *</label>
            <input type="text" name="sku" id="sku" class="form-control" required value="<?= isset($sku) ? htmlspecialchars($sku) : '' ?>">
        </div>

        <div class="mb-3">
            <label for="quantity" class="form-label">Quantity *</label>
            <input type="number" name="quantity" id="quantity" class="form-control" required min="0" value="<?= isset($quantity) ? (int)$quantity : 0 ?>">
        </div>

        <div class="mb-3">
            <label for="purchase_price" class="form-label">Purchase Price (per unit) *</label>
            <input type="number" step="0.01" name="purchase_price" id="purchase_price" class="form-control" required min="0.01" value="<?= isset($purchase_price) ? (float)$purchase_price : '' ?>">
        </div>

        <div class="mb-3">
            <label for="selling_price" class="form-label">Selling Price (per unit) *</label>
            <input type="number" step="0.01" name="selling_price" id="selling_price" class="form-control" required min="0.01" value="<?= isset($selling_price) ? (float)$selling_price : '' ?>">
        </div>

        <div class="mb-3">
            <label for="branch_id" class="form-label">Branch (optional)</label>
            <select name="branch_id" id="branch_id" class="form-select">
                <option value="">Select branch (optional)</option>
                <?php foreach ($branches as $branch): ?>
                <option value="<?= $branch['branch_id'] ?>" <?= (isset($branch_id) && $branch_id == $branch['branch_id']) ? 'selected' : '' ?>>
                    <?= htmlspecialchars($branch['branch_name']) ?>
                </option>
                <?php endforeach; ?>
            </select>
        </div>

        <button type="submit" class="btn btn-success">Add Item</button>
    </form>
</div>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
