<?php
// includes/role_helpers.php

if (session_status() == PHP_SESSION_NONE) {
    session_start();
}


/**
 * Require user to be logged in
 */
function requireLogin(): void {
    if (!isset($_SESSION['user']) && !isset($_SESSION['username'])) {
        header("Location: login.php");
        exit;
    }
}

/**
 * Check if user is logged in
 */
function isLoggedIn(): bool {
    return (isset($_SESSION['user']) || isset($_SESSION['username']));
}

/**
 * Get current logged-in user's role from session
 */
function getUserRole(): ?string {
    if (isset($_SESSION['user']['role'])) {
        return $_SESSION['user']['role'];
    }

    if (isset($_SESSION['role'])) {
        return $_SESSION['role'];
    }

    return null;
}

/**
 * Enforce role-based access control
 *
 * @param array|string $allowedRoles Role or list of roles allowed for the page
 */
function requireRole($allowedRoles): void {
    if (!isLoggedIn()) {
        header("Location: login.php");
        exit;
    }

    $userRole = getUserRole();

    if (is_string($allowedRoles)) {
        $allowedRoles = [$allowedRoles]; // convert single role to array
    }

    if (!in_array($userRole, $allowedRoles)) {
        // Unauthorized access - redirect or show error
        header("HTTP/1.1 403 Forbidden");
        echo "Access denied. You do not have permission to access this page.";
        exit;
    }
}

/**
 * Optional helper - get current logged in username
 */
function getUsername(): ?string {
    if (isset($_SESSION['user']['username'])) {
        return $_SESSION['user']['username'];
    }
    if (isset($_SESSION['username'])) {
        return $_SESSION['username'];
    }
    return null;
}
